/*
	BBitMapHandler.cpp
	
	Copyright 1996 Jeremy Moskovich, Digital Reality Studios
	All rights reserved.
	jeremy@vms.huji.ac.il
	
	A datatype for handling the Be ScreenDump file format, for use with John Watte's Datatype
	   library.
	   
	This DataType will only read in data into a DATABitmap structure, it doesn't know how to 
		create a view.
	
	It does not determine the contents of the inputted data by the type passed
	   into it, but instead by reading the first few bytes of a stream and seeing
	   if they are legal...
	   
    The format of the Be Screendump format is as follows:
    	Offset 10 - Width of image - 1
    	Offset 14 - Height of imag - 1
    	Offset 16 - color depth of image (currently one of B_RGB_32_BIT or B_COLOR_8_BIT)
    	Offset 20 - total size of bitmap data to follow
    	Offset 24 - bitMap data, with each row padded to fill longword boundries, ready
    	              to be pumped into a BBitmap file...
    	              
  	History:
  		Version 2.5:
  			-Added Be Screendump export.
  		Version 2.0:
  			-Compiled with Datatypes 1.4.0. 
  			-Identify() function now works if outType isn't specified.  This was done
  		        To allow it to work with the Version of ShowApp that comes with Datatypes 1.4.0
  		Version 1.0: Initial release.
*/

#include "DataHandler.h"
#include "Datatypes.h"
#include "DataStreams.h"

//file type of Be Screen Dump
#define SCRNDMP_TYPE	'BITM'

//number of bytes to read at once
#define BUFSIZE		2046

#define DEBUG 	0

#define SANITY( x )		if( (x) != B_NO_ERROR ) return FALSE

//============================= Function Protos =====================================
bool ScreenDmp2BBitmap( DStream & inStream, DStream & outStream );
bool BBitmap2ScreenDmp( DStream & inStream, DStream & outStream );

bool Is_Legal_Be_Screendump( DStream & inStream );
bool Is_Legal_BBitmap( DStream &inStream );
bool Do_Identify_Be_Screendump( DStream &inSpec, DATAInfo &outInfo );
bool Do_Identify_BBitmap( DStream &inSpec,DATAInfo &outInfo );

char handlerName[] = "Be ScreenDump Datatype";
long handlerVersion = 205;
char handlerInfo[] = "Version 2.5, copyright Jeremy Moskovich (jeremy@vms.huji.ac.il) all rights reserved";


/* We know how to read both BBitmaps and Be Screendumps */
Format inputFormats[] = {
	{	SCRNDMP_TYPE ,	DATA_BITMAP,
		0.9,	1.0,
		"",
		"Be Screen Dump Data"
	},
	{	DATA_BITMAP, DATA_BITMAP,
		0.9,	1.0,
		"",
		"Native Be Bitmap"
	},
	{	0,	0,
		0,	0,
		"",
		""
	}
};

/* We know how to output both BBitmaps and Be Screendumps */
Format outputFormats[] = {
	{	SCRNDMP_TYPE ,	DATA_BITMAP,
		0.9,	1.0,
		"",
		"Be Screen Dump Data"
	},
	{	DATA_BITMAP, DATA_BITMAP,
		0.9,	1.0,
		"",
		"Native Be Bitmap"
	},
	{	0,	0,
		0,	0,
		"",
		""
	}
};


/*
	This function is called by DataTypes to try and determine if we can translate to the
	specific format...
*/

long
Identify(
	DStream &			inSpec,
	const Format *		format,
	BMessage *			/* ioExtension */,
	DATAInfo &			outInfo,
	ulong				outType )
{
	#if DEBUG
		printf ("BBitMap Handler IDENTIFY Function called\n");
	#endif

//----------- We can only produce bitmap data as output
	if (outType && (outType != DATA_BITMAP) && (outType != SCRNDMP_TYPE))
	{
		#if DEBUG
			printf( "\toutType Invalid %0.4s\n", &outType  );
		#endif
		
		return DATA_NO_HANDLER;
	}

//--------- If the caller passes in a hint, then if the hint isn't equal to the 
// Be Screendump creator or a BBitmap object, we don't try to open the file...
//  If no hint is passed in, then the file type isn't taken into consideration at all
//and the file's worthiness is determined by its content alone (via the canOpen function )
	
	if( format != 0 )
	{
		if( (format->type != SCRNDMP_TYPE) && (format->type != DATA_BITMAP))
		{
			#if DEBUG
			     printf("format->type not correct : '%0.4s'\n", &format->type);
			#endif
			
			return DATA_NO_HANDLER;
		}
		else
		{
			#if DEBUG
			     printf("format->type correct!!!\n");
		    #endif
		}
	}


	//format = &inputFormats[0];

	/* Now do the actual identifying based on file content */
	if( outType == DATA_BITMAP )
		return Do_Identify_Be_Screendump( inSpec, outInfo );
		
	else if( outType == SCRNDMP_TYPE )
		return Do_Identify_BBitmap( inSpec, outInfo );
		
	else //no definitive output type specified
	{
		if( !Do_Identify_Be_Screendump( inSpec, outInfo ) )
		{
			#if DEBUG
				printf("Input is not a Be Screendump so we try to see if its a BBitmap\n");
			#endif
			return Do_Identify_BBitmap( inSpec, outInfo );
			
		}
		else //input is a Be Screendump
			return B_OK;
	}
}

bool 
Do_Identify_Be_Screendump( DStream &inSpec, DATAInfo &outInfo )
{
	
	if( Is_Legal_Be_Screendump( inSpec )  )
	{
        //Tell the other end that we're going to output BBitmap data
		outInfo.formatType = inputFormats[0].type;
		outInfo.formatGroup = inputFormats[0].t_cls;
		outInfo.formatQuality = inputFormats[0].quality;
		outInfo.formatCapability = inputFormats[0].capability;
		strcpy(outInfo.formatName, inputFormats[0].formatName);
		strcpy(outInfo.MIMEName, inputFormats[0].MIME);
		
		#if DEBUG
			printf("\tDo_Identify_Be_Screendump returned B_OK\n");
		#endif
		
		return B_OK;
	}
	else
	{
		#if DEBUG
			printf("\tDo_Identify_Be_Screendump returned DATA_NO_HANDLER\n");
		#endif
	
		return DATA_NO_HANDLER;
	}
}

bool 
Do_Identify_BBitmap( DStream &inSpec,DATAInfo &outInfo )
{
	if( Is_Legal_BBitmap( inSpec )  )
	{
        //Tell the other end that we're going to output Be Screendump data
		outInfo.formatType = inputFormats[1].type;
		outInfo.formatGroup = inputFormats[1].t_cls;
		outInfo.formatQuality = inputFormats[1].quality;
		outInfo.formatCapability = inputFormats[1].capability;
		strcpy(outInfo.formatName, inputFormats[1].formatName);
		strcpy(outInfo.MIMEName, inputFormats[1].MIME);
		
		#if DEBUG
			printf("\tDo_Identify_BBitmap returned B_OK\n");
		#endif
		
		return B_OK;
	}
	else
	{
		#if DEBUG
			printf("\tDo_Identify_BBitmap returned DATA_NO_HANDLER\n");
		#endif
		
		return DATA_NO_HANDLER;
	}
}

bool
Is_Legal_BBitmap( DStream &inStream )
{
	ulong ident;
	
	inStream.Seek( 0 );
	inStream.Read( &ident, 4 );
	
	if( ident == DATA_BITMAP )
		return TRUE;
	else
	{
		#if DEBUG
			printf("\t\tIs_Legal_BBitmap: ident != DATA_BITMAP; ident = %0.4s\n", &ident);
		#endif
		return FALSE;
	}
}

bool 
Is_Legal_Be_Screendump( DStream & inStream )
{
	ushort height, width;
	color_space bitDepth;
	ulong totalSize;

	SANITY( inStream.Seek( 10 ) );
	SANITY( inStream.Read( (void *)&width, 2 ) );
	
	SANITY( inStream.Seek( 14 ) );
	SANITY( inStream.Read( (void *)&height, 2 ) );

	SANITY( inStream.Seek( 16 ) );
	SANITY( inStream.Read( (void *)&bitDepth, 4 ) );

	SANITY( inStream.Seek( 20 ) );
	SANITY( inStream.Read( (void *)&totalSize, 4 ) );
	
	ulong rowBytes = width + 1;
	if( bitDepth == B_RGB_32_BIT )
		rowBytes *= 4;
	while( rowBytes & 3 ) rowBytes++;
	
	if( totalSize != ((height + 1) * rowBytes) )
	{
		#if DEBUG
			printf( "Total Size in file is incorrect\n" );
		#endif
		
		return FALSE;
	}
		
	if( ( width > 0 ) && (height > 0) && (bitDepth <= B_RGB_16_BIT) && (bitDepth > 0) )
	{//if theres nothing fishy about the file...
		
		#if DEBUG
		printf( "Image can be opened, dimensions are %d x %d x %d (deep) \n"
					, width, height, bitDepth );
		#endif
		
		return TRUE;
    }
    
    #if DEBUG
    	printf( "Is_Legal_Be_Screendump returning FALSE\n" );
    #endif
    return FALSE;
}

bool
ScreenDmp2BBitmap( DStream & inStream, DStream & outStream )
{
	ushort height, width;
	ulong rowBytes;
	
	color_space bitDepth;
	uchar *inBuf;
	ulong amtRead = BUFSIZE;

	ulong type = DATA_BITMAP;
	
	struct DATABitmap outSpec;
	
	#if DEBUG
		printf( "\tDoTranslate called OK\n" );
	#endif
	
	
	//Read in dimensions and butdepth from file...
	SANITY( inStream.Seek( 10 ) );
	SANITY( inStream.Read( &width, 2 ) );
	
	SANITY( inStream.Seek( 14 ) );
	inStream.Read( &height, 2 );
	
	SANITY( inStream.Seek( 16 ) );
	inStream.Read( &bitDepth, 4 );

	//calculate rowBytes
	rowBytes = width + 1;
	if( bitDepth == B_RGB_32_BIT )
		rowBytes *= 4;
	while( rowBytes & 3 ) rowBytes++;

    #if DEBUG
    	printf("Image has %d bytes per row\n", rowBytes);
    #endif

	//sanity check....
	if( ( width < 0 ) || (height < 0) || (bitDepth > B_RGB_16_BIT) || (bitDepth == 0) )
		return FALSE;
		
	#if DEBUG
		printf("\tDoTranslate: All initial Sanity checks passed OK\n");
	#endif
// F I L L  I N  DATA_BITMAP  S T R U C T U R E

	outSpec.magic = DATA_BITMAP;
	outSpec.bounds = BRect( 0, 0, width, height );
	outSpec.rowBytes = rowBytes;
	outSpec.colors = bitDepth;
	outSpec.dataSize = ((height + 1) * rowBytes);
	
	SANITY( outStream.Seek( 0 ) );
	SANITY( outStream.Write( &outSpec, sizeof(struct DATABitmap) ) );

// S T A R T  R E A D I N G  I N  B I T M A P  D A T A
	//Go to start of bitMap data...		
	SANITY( inStream.Seek( 24 ) );

	inBuf = (uchar *) malloc( BUFSIZE );
	
	if( outStream.Seek( sizeof( DATABitmap ) ) != B_NO_ERROR )
	{
		#if DEBUG
		     printf("error trying to seek to past DataBitMap, size of struct is: %d (should be %d)\n",
		       outStream.Size(), sizeof( DATABitmap ) );
		#endif
		       
		outStream.SetSize(sizeof( DATABitmap ) + 1);
		outStream.Seek( sizeof( DATABitmap ) );
	}
	
	long err = 0;
	
	do 
	{	
		err = inStream.Read(inBuf, amtRead);
		
		if( outStream.Write( inBuf, amtRead ) != B_NO_ERROR )
		{//We couldn't write all output bytes
			#if DEBUG
				printf( "\tScreenDmp2BBitmap call to Write failed\n" );
			#endif
			
			free( inBuf );
			inBuf = 0;
			return FALSE;
		}
	} while( err == B_NO_ERROR );
	
	if( inBuf ) free( inBuf );
	inBuf = 0;
	
	return TRUE;
}

bool 
BBitmap2ScreenDmp( DStream & inStream, DStream & outStream )
{
	
	//--------------- Varables --------------
	struct DATABitmap outSpec;
	ulong amtRead = BUFSIZE;
	ulong readSoFar = 0;
	ulong toCopy = 0;
	uchar *inBuf = 0;
	uchar *tmpBuf = 0;
	BRect bounds;
	ushort tmp;

	//-------------- Sanity Checks -----------
	if( !Is_Legal_BBitmap( inStream ) )
		return FALSE;

	//------------ Init --------------
	inStream.Seek( 0 );
	outStream.Seek( 0 );
	//--------- Start Translation -----------
	inStream.Read( &outSpec, sizeof( struct DATABitmap ) );

	bounds = outSpec.bounds;

	tmpBuf = (uchar *) malloc( 10 );
	if( !tmpBuf ) return FALSE; //error allocating memmory for tmpBuf
	memset( tmpBuf, 0, 10 );
	
	outStream.Write( tmpBuf, 10 );	

	tmp = bounds.Width();
	outStream.Write( &tmp, 2 );
	outStream.Write( tmpBuf, 2 );

	tmp = bounds.Height();
	outStream.Write( &tmp, 2 );
	//outStream.Write( tmpBuf, 3 );
	
	if( tmpBuf ) free( tmpBuf );
	tmpBuf = 0;
		
	outStream.Write( &outSpec.colors, 4 );
		
	toCopy = outSpec.dataSize;
	outStream.Write( &toCopy, 4 );
	
	//------------- Copy actual BitMap data ------------
	inBuf = (uchar *) malloc( BUFSIZE );
	if( !inBuf )
	{
		#if DEBUG
			printf("BBitmap2ScreenDmp: Allocation of inBuf Failed, returning false\n");
		#endif
		
		return FALSE;
	}
	
	long err = B_NO_ERROR;
	
	#if DEBUG
			printf("\tBBitmap2ScreenDmp: going into copy loop\n");
	#endif
	
	while( (readSoFar < toCopy) && (amtRead == BUFSIZE) && (err == B_NO_ERROR) )
	{	
		err = inStream.Read(inBuf, amtRead);
		
		if( outStream.Write( inBuf, amtRead ) != B_NO_ERROR )
		{//We couldn't write all output bytes
			#if DEBUG
				printf( "\tBBitmap2ScreenDmp call to Write failed\n" );
			#endif
			
			if( inBuf ) free( inBuf );
			inBuf = 0;
			return FALSE;
		}
		
		readSoFar += amtRead;
	}
	
	#if DEBUG
			printf("\tBBitmap2ScreenDmp: out of copy loop\n");
	#endif
	
	if( inBuf ) free( inBuf );
	
	return TRUE;
}

long
Translate(
	DStream &			inStream,
	const DATAInfo &	inInfo,
	BMessage *			/* ioExtension */,
	ulong				outFormat,
	DStream &			outStream)
{

	#if DEBUG
		printf ("BBitMap Handler TRANSLATE Function called\n");
		printf("outFormat = %0.4s\n", &outFormat);
    #endif
	
	//First make sure again that the file is Kosher
	if( (outFormat != DATA_BITMAP) && (outFormat != 0) && (outFormat != SCRNDMP_TYPE ) )
	{
		#if DEBUG
		     printf("\tTranslate() function failed, outFormat has an illegal value: %0.4s\n", &outFormat);
        #endif
        
		return DATA_NO_HANDLER;
	}
	

	if( outFormat == DATA_BITMAP || outFormat == 0 )
	{
		if( Is_Legal_Be_Screendump( inStream ) )
		{
			if( ScreenDmp2BBitmap( inStream, outStream ) )
			{
				#if DEBUG
				     printf("\tScreenDmp2BBitmap returned OK, Translate was succesfull\n");
				     printf("------------------------------------------------\n");
				#endif
				
				return B_OK;
			}
			else
			{
				return DATA_NO_HANDLER;
			}
		}
		else
		{
			return DATA_NO_HANDLER;
		}
	}
	else if( outFormat == SCRNDMP_TYPE )
	{ //Save as Be Screendump
		if( Is_Legal_BBitmap( inStream ) )
		{
			if( BBitmap2ScreenDmp( inStream, outStream ) )
			{
				#if DEBUG
				     printf("\tBBitmap2ScreenDmp returned OK, Translate was succesfull\n");
				     printf("------------------------------------------------\n");
				#endif	
			
				return B_OK;
			}
			else
			{
				return DATA_NO_HANDLER;
			}
		}
		else
		{
			return DATA_NO_HANDLER;
		}
	}
	else
	{//its impossible for us to get here...
		return DATA_NO_HANDLER;
	}
}